﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms.DataVisualization.Charting;

namespace aproxim
{
    public static class Approximator
    {
        public static double GetParameter(List<DataPoint> data, string operation)
        {
            double result = 0;
            int n = data.Count;
            foreach (DataPoint t in data)
            {
                if (operation == "AV_X")
                {
                    result += t.XValue/n;
                }
                if (operation == "AV_Y")
                {
                    result += t.YValues[0]/n;
                }
                if (operation == "SUM_X")
                {
                    result += t.XValue;
                }
                if (operation == "SUM_Y")
                {
                    result += t.YValues[0];
                }
                if (operation == "SUM_XY")
                {
                    result += t.XValue*t.YValues[0];
                }
                if (operation == "SUM_X2")
                {
                    result += Math.Pow(t.XValue, 2);
                }
                if (operation == "SIGMA_X")
                {
                    result += Math.Pow(t.XValue - GetParameter(data, "AV_X"), 2);
                }
                if (operation == "SIGMA_Y")
                {
                    result += Math.Pow(t.YValues[0] - GetParameter(data, "AV_Y"), 2);
                }
                if (operation == "SIGMA_XY")
                {
                    double val1 = t.XValue - GetParameter(data, "AV_X");
                    double val2 = t.YValues[0] - GetParameter(data, "AV_Y");
                    result += val1*val2;
                }
            }
            if (operation == "SIGMA_X" || operation == "SIGMA_Y" || operation == "SIGMA_XY")
            {
                result /= n;
                result = Math.Sqrt(Math.Abs(result));
            }
            return result;
        }

        public static double GetA(List<DataPoint> data)
        {
            int n = data.Count;
            double sx = GetParameter(data, "SUM_X");
            double sy = GetParameter(data, "SUM_Y");
            double sxy = GetParameter(data, "SUM_XY");
            double sx2 = GetParameter(data, "SUM_X2");
            double result = sx*sy - n*sxy;
            result /= (Math.Pow(sx, 2) - n*sx2);
            return result;
        }

        public static double GetB(List<DataPoint> data)
        {
            int n = data.Count;
            double sx = GetParameter(data, "SUM_X");
            double sy = GetParameter(data, "SUM_Y");
            double sxy = GetParameter(data, "SUM_XY");
            double sx2 = GetParameter(data, "SUM_X2");
            double result = sx*sxy - sy*sx2;
            result /= (Math.Pow(sx, 2) - n*sx2);
            return result;
        }

        public static double Delta(List<DataPoint> data1, List<DataPoint> data2)
        {
            double result = 0;
            int n = data1.Count;
            for (int index = 0; index < n; index++)
            {
                result += (100 * Math.Abs(data1[index].YValues[0] - data2[index].YValues[0])) / data1[index].YValues[0];
            }
            result /= n;
            return result;
        }

        public static double Sigma(List<DataPoint> data)
        {
            int n = data.Count;
            double a = GetA(data);
            double b = GetB(data);
            double result = data.Sum(t => Math.Pow(t.YValues[0] - b - a*t.XValue, 2));
            result /= n;
            result = Math.Sqrt(Math.Abs(result));
            return result;
        }

        public static double PolynomSigma(List<DataPoint> data, List<double> mbx, int n0)
        {
            double result = 0;
            int n = data.Count;
            foreach (DataPoint t in data)
            {
                double r = 0;
                int sign = n0 >= 0 ? 1 : -1;
                for (int idx = sign; idx != n0; idx += sign)
                {
                    r += mbx[Math.Abs(idx)]*Math.Pow(t.XValue, idx);
                }
                r += mbx[0];
                result += Math.Pow(t.YValues[0] - r, 2);
            }
            result /= n;
            result = Math.Sqrt(Math.Abs(result));
            return result;
        }

        public static double Correlation(List<DataPoint> data)
        {
            double result = Math.Pow(GetParameter(data, "SIGMA_XY"), 2);
            result /= GetParameter(data, "SIGMA_X");
            result /= GetParameter(data, "SIGMA_Y");
            return result;
        }

        public static double CorrelationRatio(List<DataPoint> linear_data, List<DataPoint> dst_data)
        {
            int n = dst_data.Count;
            double sry = GetParameter(dst_data, "AV_Y");
            double sigma2 = dst_data.Sum(t => Math.Pow(t.YValues[0] - sry, 2));
            sigma2 /= n;
            double lambda2 = linear_data.Sum(t => Math.Pow(t.YValues[0] - sry, 2));
            lambda2 /= n;
            double result = Math.Sqrt(sigma2/lambda2);
            return result;
        }

        public static bool Zeidel(double[][] a, List<double> b, List<double> mbx, int n)
        {
            for (int index = 0; index < n; index++) mbx.Add(0);
            const double e = 0.0001;
            double m;
            int iCount = 0;
            do
            {
                m = 0;
                int i;
                for (i = 0; i < n; i++)
                {
                    double s1 = 0;
                    double s2 = 0;
                    int j;
                    for (j = 0; j < i; j++) s1 += a[i][j]*mbx[j];
                    for (j = i; j < n; j++) s2 += a[i][j]*mbx[j];
                    double v = mbx[i];
                    mbx[i] -= (1 / a[i][i]) * (s1 + s2 - b[i]);
                    if (Math.Abs(v - mbx[i]) > m) m = Math.Abs(v - mbx[i]);
                }
                iCount++;
                if (iCount > 1000000) return false;
            } while (m >= e);
            return true;
        }

        public static bool GetPolynom(List<DataPoint> data, List<double> mbx, int n)
        {
            List<double> sy = new List<double> {Capacity = Math.Abs(n)};
            List<double> sx = new List<double> {Capacity = 2*Math.Abs(n) - 1};
            double[][] a = new double[n][];
            for (int index = 0; index < n; index++)
            {
                a[index] = new double[n];
            }
            mbx.Clear();
            mbx.Capacity = Math.Abs(n);
            int sign = n >= 0 ? 1 : -1;
            for (int index = 0; index != 2*n - sign; index += sign)
            {
                sx.Add(0);
                foreach (DataPoint t in data)
                {
                    sx[Math.Abs(index)] += Math.Pow(t.XValue, index);
                }
            }
            for (int index = 0; index != n; index += sign)
            {
                sy.Add(0);
                foreach (DataPoint t in data)
                {
                    sy[Math.Abs(index)] += t.YValues[0]*Math.Pow(t.XValue, index);
                }
            }
            for (int i = 0; i < Math.Abs(n); i++)
            {
                for (int j = 0; j < Math.Abs(n); j++)
                {
                    a[i][j] = sx[i + j];
                }
            }
            bool result = Zeidel(a, sy, mbx, Math.Abs(n));
            return result;
        }
    }
}
