﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Drawing;
using System.IO;
using System.Windows.Forms;
using System.Windows.Forms.DataVisualization.Charting;
using System.Xml;

namespace aproxim
{
    public partial class MainWindow : Form
    {
        private const string AverageDeviation = "Average deviation: ";
        private const string MeanSquareDeviation = "Mean-square deviation: ";
        private const string CorrelationCoefficient = "Correlation coefficient: ";
        private const string CorrelationRatio = "Correlation ratio: ";

        private double a1, b1, b2;
        private int n0;
        private readonly DataTable data;
        private readonly List<double> mbx = new List<double>();

        public MainWindow()
        {
            InitializeComponent();
            data = new DataTable("Data");
            data.Columns.Add("X", typeof(double));
            data.Columns.Add("Y", typeof(double));
            data.Columns.Add("Y0", typeof(double));
            data.Columns.Add("%", typeof(double));
            grid.DataSource = data;
            grid.Columns["Y0"].ReadOnly = true;
            grid.Columns["%"].ReadOnly = true;
            grid.AutoSizeColumnsMode = DataGridViewAutoSizeColumnsMode.Fill;
            WindowState = FormWindowState.Maximized;
            ChartArea chartArea = new ChartArea();
            chart.ChartAreas.Add(chartArea);
        }

        private void exitButton_Click(object sender, EventArgs e)
        {
            Close();
        }

        private void aboutButton_Click(object sender, EventArgs e)
        {
            new AboutBox().ShowDialog();
        }

        private void newButton_Click(object sender, EventArgs e)
        {
            data.Clear();
            // TEST DATA
            /*for (int index = -50; index <= 50; index++)
            {
                DataRow row = data.NewRow();
                row["X"] = index;
                row["Y"] = index;
                data.Rows.Add(row);
            }*/
        }

        private void openButton_Click(object sender, EventArgs e)
        {
            OpenFileDialog sdialog = new OpenFileDialog {Filter = "CSV files|*.csv|XML files|*.xml"};
            //#if WINDOWS
            sdialog.Filter += "|Microsoft Excel files|*.xls|Microsoft Excel 2007/2010 files|*.xlsx";
            //#endif
            if (sdialog.ShowDialog() == DialogResult.OK)
            {
                data.Clear();
                if (sdialog.FilterIndex == 1)
                {
                    using (StreamReader file = File.OpenText(sdialog.FileName))
                    {
                        while (!file.EndOfStream)
                        {
                            string str = file.ReadLine();
                            string[] values = str.Split(',');
                            if (values.Length == 2)
                            {
                                DataRow dataRow = data.NewRow();
                                dataRow["X"] = values[0];
                                dataRow["Y"] = values[1];
                                data.Rows.Add(dataRow);
                            }
                        }
                    }
                }
                else if (sdialog.FilterIndex == 2)
                {
                    using (FileStream file = new FileStream(sdialog.FileName, FileMode.Open))
                    {
                        using (XmlTextReader reader = new XmlTextReader(file))
                        {
                            reader.ReadStartElement();
                            while (!reader.EOF && reader.NodeType != XmlNodeType.EndElement)
                            {
                                DataRow dataRow = data.NewRow();
                                reader.ReadStartElement();
                                dataRow["X"] = reader.ReadElementContentAsDouble();
                                dataRow["Y"] = reader.ReadElementContentAsDouble();
                                reader.ReadEndElement();
                                data.Rows.Add(dataRow);
                            }
                        }
                    }
                }
                //#if WINDOWS
                else if (sdialog.FilterIndex == 3 || sdialog.FilterIndex == 4)
                {
                    Excel.OpenXls(data, sdialog.FileName);
                }
                //#endif
            }
        }

        private void saveButton_Click(object sender, EventArgs e)
        {
            SaveFileDialog sdialog = new SaveFileDialog {Filter = "CSV files|*.csv|XML files|*.xml"};
            //#if WINDOWS
            sdialog.Filter += "|Microsoft Excel files|*.xls|Microsoft Excel 2007/2010 files|*.xlsx";
            //#endif
            if (sdialog.ShowDialog() == DialogResult.OK)
            {
                if (sdialog.FilterIndex == 1)
                {
                    using (StreamWriter file = File.CreateText(sdialog.FileName))
                    {
                        List<DataPoint> src_data = FillData();
                        foreach (DataPoint point in src_data)
                        {
                            string str = point.XValue + "," + point.YValues[0];
                            file.WriteLine(str);
                        }
                    }
                }
                else if (sdialog.FilterIndex == 2)
                {
                    using (StreamWriter file = new StreamWriter(sdialog.FileName))
                    {
                        using (XmlTextWriter writer = new XmlTextWriter(file))
                        {
                            writer.WriteStartDocument();
                            writer.WriteStartElement("Data");
                            List<DataPoint> src_data = FillData();
                            foreach (DataPoint point in src_data)
                            {
                                writer.WriteStartElement("Point");
                                writer.WriteStartElement("X");
                                writer.WriteString(point.XValue.ToString());
                                writer.WriteEndElement();
                                writer.WriteStartElement("Y");
                                writer.WriteString(point.YValues[0].ToString());
                                writer.WriteEndElement();
                                writer.WriteEndElement();
                            }
                            writer.WriteEndElement();
                            writer.WriteEndDocument();
                        }
                    }
                }
                //#if WINDOWS
                else if (sdialog.FilterIndex == 3 || sdialog.FilterIndex == 4)
                {
                    Excel.SaveXls(data, sdialog.FileName);
                }
                //#endif
            }
        }

        private void runButton_Click(object sender, EventArgs e)
        {
            List<DataPoint> src_data = FillData();
            List<DataPoint> dst_data = new List<DataPoint>();
            double delta;
            double sigma;
            double r;
            int f = function.SelectedIndex + 1;
            data.Clear();
            if (f == 18 || f == 19)
            {
                n0 = (int) polynomPower.Value;
            }
            int row = Util.CalculateData(src_data, dst_data, out a1, out b1, out b2, n0, mbx, ref f);
            if (row != 0)
            {
                if (row == -1)
                {
                    MessageBox.Show(this, "Error", "Solution not found");
                }
                if (row > 0)
                {
                    MessageBox.Show(this, "Error", "Division by zero in row " + row);
                }
                return;
            }
            for (int index = 0; index < src_data.Count; index++)
            {
                DataRow dataRow = data.NewRow();
                dataRow["X"] = src_data[index].XValue;
                dataRow["Y"] = src_data[index].YValues[0];
                dataRow["Y0"] = dst_data[index].YValues[0];
                dataRow["%"] = 100 * Math.Abs(src_data[index].YValues[0] - dst_data[index].YValues[0]);
                data.Rows.Add(dataRow);
            }

            chart.Series.Clear();
            DrawData(src_data, chkPolar.CheckState == CheckState.Checked ? SeriesChartType.Polar : SeriesChartType.Point,
                     Color.Blue);
            DrawData(dst_data, chkPolar.CheckState == CheckState.Checked ? SeriesChartType.Polar : SeriesChartType.Line,
                     Color.Red);
            try
            {
                chart.ChartAreas[0].RecalculateAxesScale();
            }
            catch (Exception exc)
            {
                MessageBox.Show(exc.Message);
            }

            Util.CalculateCoefficients(src_data, dst_data, mbx, n0, out delta, out sigma, out r, f);
            TextBox[] textBoxes = new[] { txtParamA, txtParamB, txtParamB2, txtParamDelta, txtParamNu, txtParamSigma };
            foreach (TextBox textBox in textBoxes)
            {
                textBox.Text = "";
            }
            if (f <= 18 || f == 20)
            {
                txtParamA.Text = "A = " + b1;
                txtParamB.Text = "B = " + a1;
                if (f == 20)
                {
                    txtParamB2.Text = "B2 = " + b2;
                }
            }
            else if (f == 19)
            {
                string caption = string.Empty;
                for (int index = 0; index < Math.Abs(n0) + 1;index++)
                {
                    caption += "B";
                    caption += index;
                    caption += " = ";
                    caption += mbx[index];
                    caption += ";";
                }
                txtParamA.Text = caption;
            }
            txtParamDelta.Text = AverageDeviation + delta;
            txtParamSigma.Text = MeanSquareDeviation + sigma;
            if (f <= 18 || f == 20)
            {
                txtParamNu.Text = CorrelationCoefficient + r;
            }
            else if (f == 19)
            {
                txtParamNu.Text = CorrelationRatio + r;
            }
        }

        private void DrawData(IEnumerable<DataPoint> points, SeriesChartType chartType, Color color)
        {
            Series series = new Series {ChartType = chartType, Color = color};
            foreach (DataPoint point in points)
            {
                DataPoint dataPoint = new DataPoint(point.XValue, point.YValues[0]);
                series.Points.Add(dataPoint);
            }
            chart.Series.Add(series);
        }

        private List<DataPoint> FillData()
        {
            List<DataPoint> src_data = new List<DataPoint>();
            for (int index = 0; index < data.Rows.Count; index++)
            {
                DataGridViewCell item_x = grid.Rows[index].Cells[0];
                DataGridViewCell item_y = grid.Rows[index].Cells[1];
                string str_x = item_x.Value.ToString();
                string str_y = item_y.Value.ToString();
                if (!string.IsNullOrWhiteSpace(str_x) && !string.IsNullOrWhiteSpace(str_y))
                {
                    double x, y;
                    if (double.TryParse(str_x, out x) && double.TryParse(str_y, out y))
                    {
                        src_data.Add(new DataPoint(x, y));
                    }
                }
            }
            return src_data;
        }

        private void differentialButton_Click(object sender, EventArgs e)
        {
            List<DataPoint> src_data = FillData();
            for (int idx = 0; idx < src_data.Count; idx++)
            {
                double val = DataAnalysis.LagrangeD(src_data, src_data[idx].XValue);
                data.Rows[idx]["Y"] = val;
            }
        }

        private void integralButton_Click(object sender, EventArgs e)
        {
            List<DataPoint> src_data = FillData();
            for (int idx = 0; idx < src_data.Count; idx++)
            {
                double val = DataAnalysis.Simpson(0, src_data[idx].XValue, src_data);
                data.Rows[idx]["Y"] = val;
            }
        }

        private void findModelButton_Click(object sender, EventArgs e)
        {
            List<DataPoint> src_data = FillData();
            if (src_data.Count == 0) return;
            int func = Util.Epsilon(src_data);
            /*
            Epsilon = 1 -> 1
            Epsilon = 2 -> 7
            Epsilon = 3 -> 12
            Epsilon = 4 -> 9
            Epsilon = 5 -> 3
            Epsilon = 6 -> 2
            Epsilon = 7 -> 10
            */
            switch (func)
            {
                case 2:
                    func = 7;
                    break;
                case 3:
                    func = 12;
                    break;
                case 4:
                    func = 9;
                    break;
                case 5:
                    func = 3;
                    break;
                case 6:
                    func = 2;
                    break;
                case 7:
                    func = 10;
                    break;
                default:
                    func = 1;
                    break;
            }
            function.SelectedIndex = func - 1;
        }

        private void function_SelectedIndexChanged(object sender, EventArgs e)
        {
            polynomPower.Enabled = function.SelectedIndex == 17 || function.SelectedIndex == 18;
        }
    }
}
